<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\I18n\FrozenTime;
use Cake\Mailer\Mailer;
use Cake\Routing\Router;
/**
 * Users Controller
 *
 * @property \App\Model\Table\UsersTable $Users
 */
class UsersController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $query = $this->Users->find();
        $users = $this->paginate($query);

        $this->set(compact('users'));
    }

    /**
     * View method
     *
     * @param string|null $id User id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $user = $this->Users->get($id, contain: ['Carts', 'ContactSubmissions', 'Orders']);
        $this->set(compact('user'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $user = $this->Users->newEmptyEntity();

        if ($this->request->is('post')) {
            $data = (array)$this->request->getData();

            // Hash only if provided
            if (!empty($data['password'])) {
                $data['password'] = password_hash((string)$data['password'], PASSWORD_DEFAULT);
            }

            $user = $this->Users->patchEntity($user, $data);
            if ($this->Users->save($user)) {
                $this->Flash->success('The user has been saved.');
                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error('The user could not be saved. Please, try again.');
        }

        $this->set(compact('user'));
    }

    /**
     * Edit method
     *
     * @param string|null $id User id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $user = $this->Users->get((int)$id);

        if ($this->request->is(['patch', 'post', 'put'])) {
            $data = (array)$this->request->getData();

            if (array_key_exists('password', $data)) {
                if ($data['password'] === '' || $data['password'] === null) {
                    // Do not touch the password if left empty
                    unset($data['password']);
                } else {
                    // Hash the new password
                    $data['password'] = password_hash((string)$data['password'], PASSWORD_DEFAULT);
                }
            }

            $user = $this->Users->patchEntity($user, $data);
            if ($this->Users->save($user)) {
                $this->Flash->success('The user has been updated.');
                return $this->redirect(['action' => 'view', $user->id]);
            }
            $this->Flash->error('The user could not be updated. Please, try again.');
        }

        $this->set(compact('user'));
    }

    /**
     * Delete method
     *
     * @param string|null $id User id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $user = $this->Users->get($id);
        if ($this->Users->delete($user)) {
            $this->Flash->success(__('The user has been deleted.'));
        } else {
            $this->Flash->error(__('The user could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function register()
    {
        $user = $this->Users->newEmptyEntity();

        if ($this->request->is('post')) {
            $data = (array)$this->request->getData();
            $pwd  = (string)($data['password'] ?? '');
            $conf = (string)($data['password_confirm'] ?? '');

            if (strlen($pwd) < 8 || $pwd !== $conf) {
                $this->Flash->error('Passwords must match and be at least 8 characters.');
            } else {
                // prevent role escalation from the form
                unset($data['role']);

                // hash here
                $data['password'] = password_hash($pwd, PASSWORD_DEFAULT);

                $user = $this->Users->patchEntity($user, $data);
                $user->role = 'customer'; // force customer role

                if ($this->Users->save($user)) {
                    // auto-login
                    $this->request->getSession()->write('Auth.User', [
                        'id'       => $user->id,
                        'email'    => $user->email,
                        'username' => $user->username,
                        'role'     => $user->role,
                    ]);

                    $this->Flash->success('Welcome! Your account has been created.');

                    // honor ?redirect=..., else go to checkout if cart has items, else to products
                    $redirect = (string)$this->request->getQuery('redirect', '');
                    if ($redirect !== '' && strpos($redirect, '/') === 0) {
                        return $this->redirect($redirect);
                    }
                    if ($this->request->getSession()->read('Cart.items')) {
                        return $this->redirect(['controller' => 'Carts', 'action' => 'checkout']);
                    }
                    return $this->redirect(['controller' => 'Products', 'action' => 'index']);
                }

                $this->Flash->error('Could not create your account. Please fix any errors and try again.');
            }
        }

        $this->set(compact('user'));
    }

    public function login()
    {
        if ($this->request->is('post')) {
            $identifier = trim((string)$this->request->getData('identifier'));
            $password   = (string)$this->request->getData('password');

            $user = $this->Users->find()
                ->where(['OR' => [['email' => $identifier], ['username' => $identifier]]])
                ->first();

            if ($user && password_verify($password, (string)$user->password)) {
                if (password_needs_rehash((string)$user->password, PASSWORD_DEFAULT)) {
                    $user->password = password_hash($password, PASSWORD_DEFAULT);
                    $this->Users->save($user);
                }

                $this->request->getSession()->write('Auth.User', [
                    'id'       => $user->id,
                    'email'    => $user->email,
                    'username' => $user->username,
                    'role'     => $user->role,
                ]);

                $this->Flash->success('Welcome back!');

                // send them back where they came from (same-origin only)
                $redirect = (string)$this->request->getQuery('redirect', '');
                if ($redirect !== '' && strpos($redirect, '/') === 0) {
                    return $this->redirect($redirect);
                }
                return $this->redirect(['controller' => 'Pages', 'action' => 'display', 'dashboard']);
            }

            $this->Flash->error('Invalid login.');
        }
    }

    /**
     * Logout
     */
    public function logout()
    {
        $this->request->getSession()->delete('Auth.User');
        $this->Flash->success('Signed out.');
        return $this->redirect(['controller' => 'Pages', 'action' => 'home']);
    }

    /**
     * Forgot Password (send reset link)
     */
    public function forgotPassword()
    {
        if ($this->request->is('post')) {
            $email = trim((string)$this->request->getData('email'));
            $user  = $this->Users->find()->where(['email' => $email])->first();

            // Always act the same (avoid user enumeration)
            if ($user) {
                $user->password_reset_token   = bin2hex(random_bytes(32));
                $user->password_reset_expires = FrozenTime::now()->addHours(2);
                $this->Users->save($user);

                $resetUrl = Router::url(
                    ['controller' => 'Users', 'action' => 'resetPassword', $user->password_reset_token],
                    true
                );

                $mailer = new Mailer('default');
                $mailer->setTo($user->email)
                    ->setSubject('Reset your Salty & Bold password')
                    ->setEmailFormat('both')
                    ->setViewVars(['user' => $user, 'resetUrl' => $resetUrl])
                    ->viewBuilder()->setTemplate('reset_password');

                $mailer->deliver();
            }

            $this->Flash->success('If that email exists, a reset link has been sent.');
            return $this->redirect(['action' => 'login']);
        }
    }

    /**
     * Reset Password (verify token + set new hash)
     */
    public function resetPassword(string $token = '')
    {
        if ($token === '') {
            $this->Flash->error('Missing token.');
            return $this->redirect(['action' => 'login']);
        }

        $user = $this->Users->find()
            ->where([
                'password_reset_token'      => $token,
                'password_reset_expires >=' => FrozenTime::now(),
            ])->first();

        if (!$user) {
            $this->Flash->error('That reset link is invalid or has expired.');
            return $this->redirect(['action' => 'forgotPassword']);
        }

        if ($this->request->is(['post', 'put', 'patch'])) {
            $new     = (string)$this->request->getData('password');
            $confirm = (string)$this->request->getData('password_confirm');

            if (strlen($new) < 8 || $new !== $confirm) {
                $this->Flash->error('Passwords must match and be at least 8 characters.');
            } else {
                $user->password = password_hash($new, PASSWORD_DEFAULT); // hash here
                $user->password_reset_token   = null;
                $user->password_reset_expires = null;

                if ($this->Users->save($user)) {
                    $this->Flash->success('Password updated. Please sign in.');
                    return $this->redirect(['action' => 'login']);
                }
                $this->Flash->error('Could not update password. Try again.');
            }
        }

        $this->set(compact('user'));
    }
}
