<?php
declare(strict_types=1);

namespace App\Controller;

class ProductsController extends AppController
{
    public function listProducts()
    {
        $Products = $this->fetchTable('Products');

        // Read filters from query (GET)
        $seasonal = $this->getRequest()->getQuery('seasonal', ''); // '' | '1'
        $inStock  = (int)$this->getRequest()->getQuery('in_stock', 0) === 1; // 0 | 1
        $sort     = $this->getRequest()->getQuery('sort', 'newest'); // newest|price_asc|price_desc

        // Base query: all products (available or not)
        $query = $Products->find()
            ->contain(['Allergens']); // ✅ include allergens

        // Seasonal filter
        if ($seasonal === '1') {
            $query->where(['Products.is_seasonal' => 1]);
        }

        // In-stock filter
        if ($inStock) {
            $query->where(['Products.stock_level >' => 0]);
        }

        // Sorting
        switch ($sort) {
            case 'price_asc':
                $query->order(['Products.price' => 'ASC', 'Products.id' => 'DESC']);
                break;
            case 'price_desc':
                $query->order(['Products.price' => 'DESC', 'Products.id' => 'DESC']);
                break;
            case 'newest':
            default:
                $query->order(['Products.created' => 'DESC', 'Products.id' => 'DESC']);
                break;
        }

        // Paginate
        $products = $this->paginate($query, ['limit' => 12]);

        // Pass current filters for the view (to preselect + keep in pagination)
        $this->set([
            'products' => $products,
            'filters'  => [
                'seasonal' => $seasonal,
                'in_stock' => $inStock ? '1' : '0',
                'sort'     => $sort,
            ],
        ]);
    }

    public function view($id = null)
    {
        $Products = $this->fetchTable('Products');
        $product = $Products->get($id, [
            'contain' => ['Allergens'], // ✅ include allergens
        ]);
        $this->set(compact('product'));
    }

    public function add()
    {
        $product = $this->Products->newEmptyEntity();

        if ($this->request->is('post')) {
            $product = $this->Products->patchEntity($product, (array)$this->request->getData());

            if ($this->Products->save($product)) {
                $this->Flash->success('The product has been saved.');
                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error('The product could not be saved. Please, try again.');
        }

        $this->set(compact('product'));
    }

    public function index()
    {
        // Admin-facing list with pagination
        $this->paginate = [
            'limit' => 20,
            'order' => ['Products.created' => 'DESC']
        ];

        $query = $this->Products->find(); // adjust contain()/conditions if needed
        $products = $this->paginate($query);

        $this->set(compact('products'));
    }

    public function delete($id = null)
    {
        $this->request->allowMethod(['post','delete']);

        try {
            $product = $this->Products->get((int)$id);
        } catch (RecordNotFoundException $e) {
            $this->Flash->error('Product not found.');
            return $this->redirect(['action' => 'index']);
        }

        if ($this->Products->delete($product)) {
            $this->Flash->success('The product has been deleted.');
        } else {
            $this->Flash->error('The product could not be deleted. Please try again.');
        }

        return $this->redirect(['action' => 'index']);
    }

    public function edit($id = null)
    {
        try {
            $product = $this->Products->get((int)$id);
        } catch (RecordNotFoundException $e) {
            $this->Flash->error('Product not found.');
            return $this->redirect(['action' => 'index']);
        }

        if ($this->request->is(['patch','post','put'])) {
            $data = (array)$this->request->getData();

            $product = $this->Products->patchEntity($product, $data);

            if ($this->Products->save($product)) {
                $this->Flash->success('The product has been updated.');
                return $this->redirect(['action' => 'index']);
            }

            $this->Flash->error('The product could not be updated. Please, try again.');
        }

        $this->set(compact('product'));
    }

}
