<?php
declare(strict_types=1);

namespace App\Controller;
use Cake\Mailer\Mailer;
use Cake\I18n\FrozenTime;
/**
 * Carts Controller
 *
 * @property \App\Model\Table\CartsTable $Carts
 */
class CartsController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $query = $this->Carts->find()
            ->contain(['Users']);
        $carts = $this->paginate($query);

        $this->set(compact('carts'));
    }

    public function cart()
    {
        $cart = $this->getCart();
        $total = 0.0;
        foreach ($cart['items'] as $row) {
            $total += (float)$row['price'] * (int)$row['qty'];
        }
        $this->set(compact('cart','total'));
    }
    private function getCart(): array
    {
        return (array)$this->request->getSession()->read('Cart') ?: ['items' => []];
    }
    private function saveCart(array $cart): void
    {
        $this->request->getSession()->write('Cart', $cart);
    }

    /**
     * Add item (default qty 1)
     */
    public function addItem($productId = null)
    {
        $this->request->allowMethod(['post', 'get']);

        $productId = (int)($productId ?? $this->request->getData('product_id') ?? $this->request->getQuery('product_id') ?? 0);
        if ($productId <= 0) {
            $this->Flash->error('Invalid product.');
            return $this->redirect(['action' => 'cart']);
        }

        $qty = max(1, (int)$this->request->getData('qty') ?: 1);

        $Products = $this->fetchTable('Products');
        $p = $Products->find()
            ->select(['id','name','price','is_available'])
            ->where(['id' => $productId, 'is_available' => 1])
            ->first();

        if (!$p) {
            $this->Flash->error('That product is not available.');
            return $this->redirect(['action' => 'cart']);
        }

        $cart = $this->getCart();
        if (isset($cart['items'][$p->id])) {
            $cart['items'][$p->id]['qty'] += $qty;
        } else {
            $cart['items'][$p->id] = [
                'id'    => (int)$p->id,
                'name'  => (string)$p->name,
                'price' => (float)$p->price,
                'qty'   => $qty,
            ];
        }
        $this->saveCart($cart);
        $this->Flash->success('Added to cart.');
        return $this->redirect(['action' => 'cart']);
    }

    /**
     * Update quantities (POST: quantities[productId] = qty)
     */
    public function update()
    {
        $this->request->allowMethod(['post']);
        $quantities = (array)$this->request->getData('quantities', []);
        $cart = $this->getCart();

        foreach ($quantities as $pid => $q) {
            $pid = (int)$pid; $q = (int)$q;
            if (!isset($cart['items'][$pid])) { continue; }
            if ($q <= 0) {
                unset($cart['items'][$pid]);
            } else {
                $cart['items'][$pid]['qty'] = $q;
            }
        }
        $this->saveCart($cart);
        $this->Flash->success('Cart updated.');
        return $this->redirect(['action' => 'cart']);
    }

    /**
     * Remove one line
     */
    public function removeItem($productId = null)
    {
        $this->request->allowMethod(['get','post']);

        $pid = (int)($productId
            ?? $this->request->getQuery('productId')
            ?? $this->request->getData('productId')
            ?? 0);

        if ($pid <= 0) {
            $this->Flash->error('Invalid product.');
            return $this->redirect(['action' => 'cart']);
        }

        $cart = $this->getCart();
        if (isset($cart['items'][$pid])) {
            unset($cart['items'][$pid]);
            $this->saveCart($cart);
            $this->Flash->success('Item removed.');
        } else {
            $this->Flash->error('Item not found in cart.');
        }

        return $this->redirect(['action' => 'cart']);
    }

    /**
     * Clear cart
     */
    public function clear()
    {
        $this->request->allowMethod(['post']);
        $this->request->getSession()->delete('Cart');
        $this->Flash->success('Cart cleared.');
        return $this->redirect(['action' => 'cart']);
    }

    /**
     * Checkout (GET shows form, POST emails order summary to Mailtrap)
     */
    public function checkout()
    {
        $cart = $this->getCart();
        if (empty($cart['items'])) {
            $this->Flash->error('Your cart is empty.');
            return $this->redirect(['action' => 'cart']);
        }

        if ($this->request->is('post')) {
            $d = (array)$this->request->getData();

            // minimal validation
            $name  = trim((string)($d['name']  ?? ''));
            $email = trim((string)($d['email'] ?? ''));
            $addr  = trim((string)($d['address'] ?? ''));
            $phone = trim((string)($d['phone'] ?? ''));
            $note  = trim((string)($d['notes'] ?? ''));

            if ($name === '' || !filter_var($email, FILTER_VALIDATE_EMAIL) || $addr === '') {
                $this->Flash->error('Please provide name, a valid email, and address.');
                return $this->redirect(['action' => 'checkout']);
            }

            // compute totals
            $total = 0.0;
            foreach ($cart['items'] as $row) {
                $total += (float)$row['price'] * (int)$row['qty'];
            }

            // send email
            $mailer = new Mailer('default');
            $mailer->setTo($email)
                ->setSubject('Your Salty & Bold order (dev)')
                ->setEmailFormat('both')
                ->setViewVars([
                    'customer' => compact('name','email','phone','addr','note'),
                    'items'    => array_values($cart['items']),
                    'total'    => $total,
                    'placedAt' => FrozenTime::now(),
                ])
                ->viewBuilder()->setTemplate('order_summary');

            $mailer->deliver();

            // clear cart
            $this->request->getSession()->delete('Cart');
            $this->Flash->success('Order placed! A summary was emailed to you.');
            return $this->redirect(['controller' => 'Pages', 'action' => 'home']);
        }

        // GET
        $total = 0.0;
        foreach ($cart['items'] as $row) {
            $total += (float)$row['price'] * (int)$row['qty'];
        }
        $this->set(compact('cart','total'));
    }

    /**
     * View method
     *
     * @param string|null $id Cart id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $cart = $this->Carts->get($id, contain: ['Users', 'CartItems']);
        $this->set(compact('cart'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $cart = $this->Carts->newEmptyEntity();
        if ($this->request->is('post')) {
            $cart = $this->Carts->patchEntity($cart, $this->request->getData());
            if ($this->Carts->save($cart)) {
                $this->Flash->success(__('The cart has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The cart could not be saved. Please, try again.'));
        }
        $users = $this->Carts->Users->find('list', limit: 200)->all();
        $this->set(compact('cart', 'users'));
    }

    /**
     * Edit method
     *
     * @param string|null $id Cart id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $cart = $this->Carts->get($id, contain: []);
        if ($this->request->is(['patch', 'post', 'put'])) {
            $cart = $this->Carts->patchEntity($cart, $this->request->getData());
            if ($this->Carts->save($cart)) {
                $this->Flash->success(__('The cart has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The cart could not be saved. Please, try again.'));
        }
        $users = $this->Carts->Users->find('list', limit: 200)->all();
        $this->set(compact('cart', 'users'));
    }

    /**
     * Delete method
     *
     * @param string|null $id Cart id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $cart = $this->Carts->get($id);
        if ($this->Carts->delete($cart)) {
            $this->Flash->success(__('The cart has been deleted.'));
        } else {
            $this->Flash->error(__('The cart could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }
}
